#include "colormodels.h"
#include <glib.h>
#include <stdlib.h>


cmodel_yuv_t *yuv_table = 0;

void cmodel_init_yuv(cmodel_yuv_t *yuv_table)
{
	int i;
	for(i = 0; i < 256; i++)
	{
/* compression */
		yuv_table->rtoy_tab[i] = (long)( 0.2990 * 65536 * i);
		yuv_table->rtou_tab[i] = (long)(-0.1687 * 65536 * i);
		yuv_table->rtov_tab[i] = (long)( 0.5000 * 65536 * i);

		yuv_table->gtoy_tab[i] = (long)( 0.5870 * 65536 * i);
		yuv_table->gtou_tab[i] = (long)(-0.3320 * 65536 * i);
		yuv_table->gtov_tab[i] = (long)(-0.4187 * 65536 * i);

		yuv_table->btoy_tab[i] = (long)( 0.1140 * 65536 * i);
		yuv_table->btou_tab[i] = (long)( 0.5000 * 65536 * i) + 0x800000;
		yuv_table->btov_tab[i] = (long)(-0.0813 * 65536 * i) + 0x800000;
	}

	yuv_table->vtor = &(yuv_table->vtor_tab[128]);
	yuv_table->vtog = &(yuv_table->vtog_tab[128]);
	yuv_table->utog = &(yuv_table->utog_tab[128]);
	yuv_table->utob = &(yuv_table->utob_tab[128]);
	for(i = -128; i < 128; i++)
	{
/* decompression */
		yuv_table->vtor[i] = (long)( 1.4020 * 65536 * i);
		yuv_table->vtog[i] = (long)(-0.7141 * 65536 * i);

		yuv_table->utog[i] = (long)(-0.3441 * 65536 * i);
		yuv_table->utob[i] = (long)( 1.7720 * 65536 * i);
	}
}


void cmodel_delete_yuv(cmodel_yuv_t *yuv_table)
{
}

int cmodel_is_planar(int colormodel)
{
	switch(colormodel)
	{
		case BC_YUV420P:      return 1; break;
		case BC_YUV422P:      return 1; break;
		case BC_YUV411P:      return 1; break;
	}
	return 0;
}

int cmodel_calculate_pixelsize(int colormodel)
{
	switch(colormodel)
	{
		case BC_TRANSPARENCY: return 1; break;
		case BC_COMPRESSED:   return 1; break;
		case BC_RGB8:         return 1; break;
		case BC_RGB565:       return 2; break;
		case BC_BGR565:       return 2; break;
		case BC_BGR888:       return 3; break;
		case BC_BGR8888:      return 4; break;
// Working bitmaps are packed to simplify processing
		case BC_RGB888:       return 3; break;
		case BC_RGBA8888:     return 4; break;
		case BC_RGB161616:    return 6; break;
		case BC_RGBA16161616: return 8; break;
		case BC_YUV888:       return 3; break;
		case BC_YUVA8888:     return 4; break;
		case BC_YUV161616:    return 6; break;
		case BC_YUVA16161616: return 8; break;
// Planar
		case BC_YUV420P:      return 1; break;
		case BC_YUV422P:      return 1; break;
		case BC_YUV422:       return 2; break;
		case BC_YUV411P:      return 1; break;
	}
	return 0;
}

int cmodel_calculate_datasize(int w, int h, int bytes_per_line, int color_model)
{
	if(bytes_per_line < 0) bytes_per_line = w * cmodel_calculate_pixelsize(color_model);
	switch(color_model)
	{
		case BC_YUV420P:
		case BC_YUV411P:
			return w * h + w * h / 2 + 4;
			break;

		case BC_YUV422P:
			return w * h * 2 + 4;
			break;

		default:
			return h * bytes_per_line + 4;
			break;
	}
	return 0;
}


static void get_scale_tables(int **column_table, 
	int **row_table, 
	int in_x1, 
	int in_y1, 
	int in_x2, 
	int in_y2,
	int out_x1, 
	int out_y1, 
	int out_x2, 
	int out_y2)
{
	int y_out, i;
	float w_in = in_x2 - in_x1;
	float h_in = in_y2 - in_y1;
	int w_out = out_x2 - out_x1;
	int h_out = out_y2 - out_y1;

	float hscale = w_in / w_out;
	float vscale = h_in / h_out;

	(*column_table) = malloc(sizeof(int) * w_out);
	(*row_table) = malloc(sizeof(int) * h_out);
	for(i = 0; i < w_out; i++)
	{
		(*column_table)[i] = (int)(hscale * i) + in_x1;
	}

	for(i = 0; i < h_out; i++)
	{
		(*row_table)[i] = (int)(vscale * i) + in_y1;
//printf("get_scale_tables %d %d\n", i, (int)(vscale * i));
	}
}

void cmodel_transfer(unsigned char **output_rows, 
	unsigned char **input_rows,
	unsigned char *out_y_plane,
	unsigned char *out_u_plane,
	unsigned char *out_v_plane,
	unsigned char *in_y_plane,
	unsigned char *in_u_plane,
	unsigned char *in_v_plane,
	int in_x, 
	int in_y, 
	int in_w, 
	int in_h,
	int out_x, 
	int out_y, 
	int out_w, 
	int out_h,
	int in_colormodel, 
	int out_colormodel,
	int bg_color,
	int total_in_w,
	int total_out_w)
{
	int *column_table;
	int *row_table;
	int scale_x, scale;
	int bg_r, bg_g, bg_b;
	int in_pixelsize = cmodel_calculate_pixelsize(in_colormodel);
	int out_pixelsize = cmodel_calculate_pixelsize(out_colormodel);

	bg_r = (bg_color & 0xff0000) >> 16;
	bg_g = (bg_color & 0xff00) >> 8;
	bg_b = (bg_color & 0xff);

// Initialize tables
	if(yuv_table == 0)
	{
		yuv_table = calloc(1, sizeof(cmodel_yuv_t));
		cmodel_init_yuv(yuv_table);
	}

// Get scaling
	scale = scale_x = (out_w != in_w);
	get_scale_tables(&column_table, &row_table, 
		in_x, in_y, in_x + in_w, in_y + in_h,
		out_x, out_y, out_x + out_w, out_y + out_h);

//printf("cmodel_transfer %d %d %d %d %d %d\n", in_colormodel, out_colormodel,
// out_x, out_y, out_w, out_h);

	switch(in_colormodel)
	{
		case BC_YUV420P:
		case BC_YUV422P:
			cmodel_yuv420p(output_rows,  \
				input_rows, \
				out_y_plane, \
				out_u_plane, \
				out_v_plane, \
				in_y_plane, \
				in_u_plane, \
				in_v_plane, \
				in_x,  \
				in_y,  \
				in_w,  \
				in_h, \
				out_x,  \
				out_y,  \
				out_w,  \
				out_h, \
				in_colormodel,  \
				out_colormodel, \
				bg_color, \
				total_in_w, \
				total_out_w, \
				scale, \
				out_pixelsize, \
				in_pixelsize, \
				row_table, \
				column_table, \
				bg_r, \
				bg_g, \
				bg_b);
			break;

		case BC_YUV422:
			cmodel_yuv422(output_rows,  \
				input_rows, \
				out_y_plane, \
				out_u_plane, \
				out_v_plane, \
				in_y_plane, \
				in_u_plane, \
				in_v_plane, \
				in_x,  \
				in_y,  \
				in_w,  \
				in_h, \
				out_x,  \
				out_y,  \
				out_w,  \
				out_h, \
				in_colormodel,  \
				out_colormodel, \
				bg_color, \
				total_in_w, \
				total_out_w, \
				scale, \
				out_pixelsize, \
				in_pixelsize, \
				row_table, \
				column_table, \
				bg_r, \
				bg_g, \
				bg_b);
			break;

		default:
			cmodel_default(output_rows,  \
				input_rows, \
				out_y_plane, \
				out_u_plane, \
				out_v_plane, \
				in_y_plane, \
				in_u_plane, \
				in_v_plane, \
				in_x,  \
				in_y,  \
				in_w,  \
				in_h, \
				out_x,  \
				out_y,  \
				out_w,  \
				out_h, \
				in_colormodel,  \
				out_colormodel, \
				bg_color, \
				total_in_w, \
				total_out_w, \
				scale, \
				out_pixelsize, \
				in_pixelsize, \
				row_table, \
				column_table, \
				bg_r, \
				bg_g, \
				bg_b);
			break;
	}

	free(column_table);
	free(row_table);
}

int cmodel_bc_to_x(int color_model)
{
	switch(color_model)
	{
		case BC_YUV420P:
			return FOURCC_YV12;
			break;
		case BC_YUV422:
			return FOURCC_YUV2;
			break;
	}
	return -1;
}

void cmodel_to_text(char *string, int cmodel)
{
	switch(cmodel)
	{
		case BC_RGB888:       strcpy(string, "RGB-8 Bit");   break;
		case BC_RGBA8888:     strcpy(string, "RGBA-8 Bit");  break;
		case BC_RGB161616:    strcpy(string, "RGB-16 Bit");  break;
		case BC_RGBA16161616: strcpy(string, "RGBA-16 Bit"); break;
		case BC_YUV888:       strcpy(string, "YUV-8 Bit");   break;
		case BC_YUVA8888:     strcpy(string, "YUVA-8 Bit");  break;
		case BC_YUV161616:    strcpy(string, "YUV-16 Bit");  break;
		case BC_YUVA16161616: strcpy(string, "YUVA-16 Bit"); break;
		default: strcpy(string, "RGB-8 Bit"); break;
	}
}

int cmodel_from_text(char *text)
{
	if(!strcasecmp(text, "RGB-8 Bit"))   return BC_RGB888;
	if(!strcasecmp(text, "RGBA-8 Bit"))  return BC_RGBA8888;
	if(!strcasecmp(text, "RGB-16 Bit"))  return BC_RGB161616;
	if(!strcasecmp(text, "RGBA-16 Bit")) return BC_RGBA16161616;
	if(!strcasecmp(text, "YUV-8 Bit"))   return BC_YUV888;
	if(!strcasecmp(text, "YUVA-8 Bit"))  return BC_YUVA8888;
	if(!strcasecmp(text, "YUV-16 Bit"))  return BC_YUV161616;
	if(!strcasecmp(text, "YUVA-16 Bit")) return BC_YUVA16161616;
	return BC_RGB888;
}

