// DO NOT USE stdio.h!  printf() calls malloc()!
//#include <stdio.h>
#include <stdarg.h>
#include <crtdbg.h>

#include <windows.h>

#include "resource.h"
#include "disasm.h"
#include "oshelper.h"
#include "helpfile.h"

///////////////////////////////////////////////////////////////////////////

#define CODE_WINDOW (256)

///////////////////////////////////////////////////////////////////////////

extern HINSTANCE g_hInst;
extern "C" unsigned long version_num;

static CodeDisassemblyWindow *g_pcdw;

///////////////////////////////////////////////////////////////////////////

BOOL APIENTRY CrashDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam);

///////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG

void checkfpustack(const char *file, const int line) throw() {
	static const char szFPUProblemCaption[]="FPU/MMX internal problem";
	static const char szFPUProblemMessage[]="The FPU stack wasn't empty!  Tagword = %04x\nFile: %s, line %d";
	static bool seenmsg=false;

	char	buf[128];
	unsigned short tagword;

	if (seenmsg)
		return;

	__asm fnstenv buf

	tagword = *(unsigned short *)(buf + 8);

	if (tagword != 0xffff) {
		wsprintf(buf, szFPUProblemMessage, tagword, file, line);
		MessageBox(NULL, buf, szFPUProblemCaption, MB_OK);
		seenmsg=true;
	}

}

void __declspec(naked) *operator new(size_t bytes) {
	static const char fname[]="stack trace";

	__asm {
		push	ebp
		mov		ebp,esp

		push	[ebp+4]				;return address
		push	offset fname		;'filename'
		push	_NORMAL_BLOCK		;block type
		push	[ebp+8]				;allocation size

		call	_malloc_dbg
		add		esp,16

		pop		ebp
		ret
	}
}

#endif

#if 0
void __declspec(naked) stackcheck(void *&sp) {
	static const char g_szStackHemorrhage[]="WARNING: Thread is hemorrhaging stack space!\n";

	__asm {
		mov		eax,[esp+4]
		mov		ecx,[eax]
		or		ecx,ecx
		jnz		started
		mov		[eax],esp
		ret
started:
		sub		ecx,esp
		mov		eax,ecx
		sar		ecx,31
		xor		eax,ecx
		sub		eax,ecx
		cmp		eax,128
		jb		ok
		push	offset g_szStackHemorrhage
		call	dword ptr [OutputDebugString]
		int		3
ok:
		ret
	}
}
#endif

///////////////////////////////////////////////////////////////////////////

static const struct ExceptionLookup {
	DWORD	code;
	const char *name;
} exceptions[]={
	{	EXCEPTION_ACCESS_VIOLATION,			"Access Violation"		},
	{	EXCEPTION_BREAKPOINT,				"Breakpoint"			},
	{	EXCEPTION_FLT_DENORMAL_OPERAND,		"FP Denormal Operand"	},
	{	EXCEPTION_FLT_DIVIDE_BY_ZERO,		"FP Divide-by-Zero"		},
	{	EXCEPTION_FLT_INEXACT_RESULT,		"FP Inexact Result"		},
	{	EXCEPTION_FLT_INVALID_OPERATION,	"FP Invalid Operation"	},
	{	EXCEPTION_FLT_OVERFLOW,				"FP Overflow",			},
	{	EXCEPTION_FLT_STACK_CHECK,			"FP Stack Check",		},
	{	EXCEPTION_FLT_UNDERFLOW,			"FP Underflow",			},
	{	EXCEPTION_INT_DIVIDE_BY_ZERO,		"Integer Divide-by-Zero",	},
	{	EXCEPTION_INT_OVERFLOW,				"Integer Overflow",		},
	{	EXCEPTION_PRIV_INSTRUCTION,			"Privileged Instruction",	},
	{	EXCEPTION_ILLEGAL_INSTRUCTION,		"Illegal instruction"	},
	{	0xe06d7363,							"Unhandled Microsoft C++ Exception",	},
			// hmm... '_msc'... gee, who would have thought?
	{	NULL	},
};

LONG __stdcall CrashHandler(EXCEPTION_POINTERS *pExc) {
	SetUnhandledExceptionFilter(NULL);

	static char buf[CODE_WINDOW+16];
	HANDLE hprMe = GetCurrentProcess();
	void *lpBaseAddress = pExc->ExceptionRecord->ExceptionAddress;
	char *lpAddr = (char *)((long)lpBaseAddress & -32);

	memset(buf, 0, sizeof buf);

	if ((long)lpAddr > CODE_WINDOW/2)
		lpAddr -= CODE_WINDOW/2;
	else
		lpAddr = NULL;

	if (!ReadProcessMemory(hprMe, lpAddr, buf, CODE_WINDOW, NULL)) {
		int i;

		for(i=0; i<CODE_WINDOW; i+=32)
			if (!ReadProcessMemory(hprMe, lpAddr+i, buf+i, 32, NULL))
				memset(lpAddr+i, 0, 32);
	}

	CodeDisassemblyWindow cdw(buf, CODE_WINDOW, (char *)(buf-lpAddr), lpAddr);

	g_pcdw = &cdw;

	DialogBoxParam(g_hInst, MAKEINTRESOURCE(IDD_DISASM_CRASH), NULL, CrashDlgProc, (LPARAM)pExc);

	UnhandledExceptionFilter(pExc);

	return EXCEPTION_EXECUTE_HANDLER;
}

static void Report(HWND hwndList, HANDLE hFile, const char *format, ...) {
	char buf[256];
	va_list val;
	int ch;

	va_start(val, format);
	ch = wvsprintf(buf, format, val);
	va_end(val);

	if (hwndList)
		SendMessage(hwndList, LB_ADDSTRING, 0, (LPARAM)buf);

	if (hFile) {
		DWORD dwActual;

		buf[ch] = '\r';
		buf[ch+1] = '\n';
		WriteFile(hFile, buf, ch+2, &dwActual, NULL);
		FlushFileBuffers(hFile);
	}
}

static void SetWindowTitlef(HWND hwnd, const char *format, ...) {
	char buf[256];
	va_list val;

	va_start(val, format);
	wvsprintf(buf, format, val);
	va_end(val);
	SetWindowText(hwnd, buf);
}

static void ReportCrashData(HWND hwnd, HWND hwndReason, HANDLE hFile, const EXCEPTION_POINTERS *const pExc) {
	const EXCEPTION_RECORD *const pRecord = (const EXCEPTION_RECORD *)pExc->ExceptionRecord;
	const CONTEXT *const pContext = (const CONTEXT *)pExc->ContextRecord;
	int i, tos;

	Report(hwnd, hFile, "EAX = %08lx", pContext->Eax);
	Report(hwnd, hFile, "EBX = %08lx", pContext->Ebx);
	Report(hwnd, hFile, "ECX = %08lx", pContext->Ecx);
	Report(hwnd, hFile, "EDX = %08lx", pContext->Edx);
	Report(hwnd, hFile, "EBP = %08lx", pContext->Ebp);
	Report(hwnd, hFile, "DS:ESI = %04x:%08lx", pContext->SegDs, pContext->Esi);
	Report(hwnd, hFile, "ES:EDI = %04x:%08lx", pContext->SegEs, pContext->Edi);
	Report(hwnd, hFile, "SS:ESP = %04x:%08lx", pContext->SegSs, pContext->Esp);
	Report(hwnd, hFile, "CS:EIP = %04x:%08lx", pContext->SegCs, pContext->Eip);
	Report(hwnd, hFile, "FS = %04x", pContext->SegFs);
	Report(hwnd, hFile, "GS = %04x", pContext->SegGs);
	Report(hwnd, hFile, "EFLAGS = %08lx", pContext->EFlags);
	Report(hwnd, hFile, "");

	// extract out MMX registers

	tos = (pContext->FloatSave.StatusWord & 0x3800)>>11;

	for(i=0; i<8; i++) {
		long *pReg = (long *)(pContext->FloatSave.RegisterArea + 10*((i-tos) & 7));

		Report(hwnd, hFile, "MM%c = %08lx%08lx", i+'0', pReg[0], pReg[1]);
	}

	// fill out bomb reason

	const struct ExceptionLookup *pel = exceptions;

	while(pel->code) {
		if (pel->code == pRecord->ExceptionCode)
			break;

		++pel;
	}

	// Unfortunately, EXCEPTION_ACCESS_VIOLATION doesn't seem to provide
	// us with the read/write flag and virtual address as the docs say...
	// *sigh*

	if (!pel->code) {
		if (hwndReason)
			SetWindowTitlef(hwndReason, "Crash reason: unknown exception 0x%08lx", pRecord->ExceptionCode);

		if (hFile)
			Report(NULL, hFile, "Crash reason: unknown exception 0x%08lx", pRecord->ExceptionCode);
	} else {
		if (hwndReason)
			SetWindowTitlef(hwndReason, "Crash reason: %s", pel->name);

		if (hFile)
			Report(NULL, hFile, "Crash reason: %s", pel->name);
	}

}

static const char *GetNameFromHeap(const char *heap, int idx) {
	while(idx--)
		while(*heap++);

	return heap;
}

static void SpliceProgramPath(char *buf, int bufsiz, const char *fn) {
	char tbuf[MAX_PATH];
	char *pszFile;

	GetModuleFileName(NULL, tbuf, sizeof tbuf);
	GetFullPathName(tbuf, bufsiz, buf, &pszFile);
	strcpy(pszFile, fn);
}

static void ReportCrashCallStack(HWND hwnd, HANDLE hFile, const EXCEPTION_POINTERS *const pExc) {
	const CONTEXT *const pContext = (const CONTEXT *)pExc->ContextRecord;
	HANDLE hprMe = GetCurrentProcess();
	char *lpAddr = (char *)pContext->Esp;
	int limit = 100;
	unsigned long data, first_rva;
	const char *debug_data = NULL;
	const char *fnname_heap, *classname_heap, *rva_heap;
	const unsigned long (*seg_heap)[2];
	int seg_cnt;

	// Attempt to read debug file.

	{
		char szPath[MAX_PATH];

		SpliceProgramPath(szPath, sizeof szPath, "VirtualDub.dbg");

		HANDLE hFile2 = CreateFile(szPath, GENERIC_READ, 0, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
		bool fSuccessful = false;
		LONG lFileSize;
		DWORD dwActual;

		do {
			if (INVALID_HANDLE_VALUE == hFile2)
				break;

			lFileSize = GetFileSize(hFile2, NULL);

			if (0xFFFFFFFF == lFileSize)
				break;

			debug_data = (const char *)VirtualAlloc(NULL, lFileSize, MEM_COMMIT, PAGE_READWRITE);
			if (!debug_data)
				break;

			if (!ReadFile(hFile2, (void *)debug_data, lFileSize, &dwActual, NULL) || dwActual!=lFileSize)
				break;

			fSuccessful = true;
		} while(0);

		if (hFile2 != INVALID_HANDLE_VALUE)
			CloseHandle(hFile2);

		if (!fSuccessful) {
			if (debug_data)
				VirtualFree((void *)debug_data, 0, MEM_RELEASE);

			Report(hwnd, hFile, "Could not open debug resource file.");
			return;
		}

		rva_heap = debug_data + 20;
		classname_heap = rva_heap + ((long *)debug_data)[1];
		fnname_heap = classname_heap + ((long *)debug_data)[2];
		seg_heap = (unsigned long (*)[2])(fnname_heap + ((long *)debug_data)[3]);
		seg_cnt = ((long *)debug_data)[4];

		first_rva = *(long *)rva_heap;
		rva_heap += 4;
	}

	// Walk up the stack.  Hopefully it wasn't fscked.

	if (*(long *)debug_data != version_num) {
		Report(hwnd, hFile, "Wrong debug resource file (build %d)", *(long *)debug_data);
	} else {
		data = pContext->Eip;
		do {
			int i;

			for(i=0; i<seg_cnt; i++)
				if (data >= seg_heap[i][0] && data < seg_heap[i][0] + seg_heap[i][1])
					break;

			if (i<seg_cnt) {
				int idx = -1;
				const char *pp = rva_heap;
				long rva = data;
				long diff = 0;

				// scan down the RVAs

				rva -= first_rva;

				while(rva >= 0 && pp<classname_heap) {
					char c;

					diff = 0;
					do {
						c = *pp++;

						diff = (diff<<7) | (c & 0x7f);
					} while(c & 0x80);

					rva -= diff;
					++idx;
				}

				if (pp<classname_heap && idx>=0) {
					const char *fn_name = GetNameFromHeap(fnname_heap, idx);
					const char *class_name = NULL;
					const char *prefix = "";

					if (*fn_name < 32) {
						int class_idx;

						class_idx = ((unsigned)fn_name[0] - 1)*128 + ((unsigned)fn_name[1] - 1);
						class_name = GetNameFromHeap(classname_heap, class_idx);

						fn_name += 2;

						if (*fn_name == 1) {
							fn_name = class_name;
						} else if (*fn_name == 2) {
							fn_name = class_name;
							prefix = "~";
						} else if (*fn_name < 32)
							fn_name = "(special)";
					}

					Report(hwnd, hFile, "%08lx: %s%s%s%s()", data, class_name?class_name:"", class_name?"::":"", prefix, fn_name);
				}/* else
					Report(hwnd, hFile, "%08lx %d", data, idx);*/

				// still room?

				if (!--limit)
					break;
			}

			lpAddr += 4;
		} while(ReadProcessMemory(hprMe, lpAddr-4, &data, 4, NULL));
	}

	VirtualFree((void *)debug_data, 0, MEM_RELEASE);
}

void DoSave(const EXCEPTION_POINTERS *pExc) {
	HANDLE hFile;
	char szModName2[MAX_PATH];
	char tbuf[256];
	long idx;

	SpliceProgramPath(szModName2, sizeof szModName2, "crashinfo.txt");

	hFile = CreateFile(szModName2, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);

	if (INVALID_HANDLE_VALUE == hFile)
		return;

	Report(NULL, hFile,
			"VirtualDub crash report -- build %d\r\n"
			"-------------------------------------\r\n"
			"\r\n"
			"Disassembly:", version_num);

	idx = 0;

	while(idx = g_pcdw->getInstruction(tbuf, idx)) {
		Report(NULL, hFile, "%s", tbuf);
	}

	Report(NULL, hFile, "");

	ReportCrashData(NULL, NULL, hFile, pExc);

	Report(NULL, hFile, "");

	ReportCrashCallStack(NULL, hFile, pExc);

	Report(NULL, hFile, "\n-- End of report");

	CloseHandle(hFile);
}

void DoHelp(HWND hwnd) {
	char buf[512];

	strcpy(buf, HelpGetPath());
	strcat(buf, ">Helpme");

	WinHelp(hwnd, buf, HELP_CONTEXT, IDH_CRASH);
}

BOOL APIENTRY CrashDlgProc(HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam) {
	static const EXCEPTION_POINTERS *s_pExc;

	switch(msg) {

		case WM_INITDIALOG:
			{
				HWND hwndList1 = GetDlgItem(hDlg, IDC_ASMBOX);
				HWND hwndList2 = GetDlgItem(hDlg, IDC_REGDUMP);
				HWND hwndList3 = GetDlgItem(hDlg, IDC_CALL_STACK);
				HWND hwndReason = GetDlgItem(hDlg, IDC_STATIC_BOMBREASON);
				const EXCEPTION_POINTERS *const pExc = (const EXCEPTION_POINTERS *)lParam;
				const EXCEPTION_RECORD *const pRecord = (const EXCEPTION_RECORD *)pExc->ExceptionRecord;
				const CONTEXT *const pContext = (const CONTEXT *)pExc->ContextRecord;

				s_pExc = pExc;

				g_pcdw->DoInitListbox(hwndList1);

				SendMessage(hwndList2, WM_SETFONT, SendMessage(hwndList1, WM_GETFONT, 0, 0), MAKELPARAM(TRUE, 0));
				SendMessage(hwndList3, WM_SETFONT, SendMessage(hwndList1, WM_GETFONT, 0, 0), MAKELPARAM(TRUE, 0));

				ReportCrashData(hwndList2, hwndReason, NULL, pExc);
				ReportCrashCallStack(hwndList3, NULL, pExc);

			}
			return TRUE;

		case WM_COMMAND:
			switch(LOWORD(wParam)) {
			case IDCANCEL: case IDOK:
				EndDialog(hDlg, FALSE);
				return TRUE;
			case IDC_SAVE2:
				DoSave(s_pExc);
				return TRUE;
			case IDC_HELP2:
				DoHelp(hDlg);
				return TRUE;
			}
			break;

		case WM_MEASUREITEM:
			return g_pcdw->DoMeasureItem(lParam);

		case WM_DRAWITEM:
			return g_pcdw->DoDrawItem(lParam);
	}

	return FALSE;
}
